<?php

namespace Accounting\config;

use Accounting\ParamSetup;
use Base\DBQueries;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class OperationsTagging extends ParamSetup
{
    private string $tbl = 'tbl_operations_relationship';
    private Request $request;
    private ?array $others;
    private bool $isInSubsidiaryLedger;

    public function setParams()
    {
        return ['request' => $this->request, 'table' => $this->tbl];
    }

    public function main (Request $request) : JsonResponse
    {
        $this->request = $request;
        if($request->type === 'code') {
            $this->conditions = $request->conditions;
            $this->retrieveFromSubsidiaryLedger();
            return successResponse("msg",$this->isInSubsidiaryLedger);
        }
        try {
            if ($this->generateParams()) {
                $this->filterTable = $this->table;
                $action = $this->action;
                if (in_array($this->filterTable, array_keys(HIDDEN_REQUEST_FIELDS))) $this->others = $this->filterRequest();
                if($action === 'dbInsert') $this->insertOperationRelationship();
                else if($action === 'dbDelete') $this->removeOperations();
                else $this->execQuery();

                return $this->getResponse($action);
            }
        } catch (\Exception $ex){
            return failedResponse("ERROR", ["msg" => $ex->getMessage()], 422);
        }
        return failedResponse("ERROR", ["msg" => "Unknown Error!"], 400);
    }

    private function insertOperationRelationship() : void
    {
        $operation = $this->getValue('operation_code');
        $branch = $this->getValue('branch_code');
        $department = $this->getValue('department_code');

        $operation_name = $this->getOperationName($operation);
        $branch_name = $this->getOperationName($branch);
        $department_name = '';
        if($department) $department_name = $this->getOperationName($department);

        $code_array = array($operation, $branch, $department);
        $name_array = array($operation_name, $branch_name, $department_name);

        $code = $operation . $branch;
        $parent_name = $operation_name . ' - ' . $branch_name;
        $type = 'branch';

        $isExisting = $this->isExistingToTable($code);

        if($isExisting === 0) {
            $this->addToRelationshipTable($code, $branch, $operation, $type, $parent_name);
            $this->updateNode($operation);
        }
        else if ($isExisting === -1) {
            $this->setDeletedAtToNull($code);
            $this->updateNode($operation);
        }

        $code = implode('', $code_array);
        $parent_code = $operation . $branch;
        $desc = implode(' - ', $name_array);

        $type = 'department';

        $isExisting = $this->isExistingToTable($code);
        if($isExisting === 0) {
            $this->addToRelationshipTable($code, $department, $parent_code, $type, $desc);
            $this->updateNode($parent_code);
        }
        else if ($isExisting === -1) {
            $this->setDeletedAtToNull($code);
            $this->updateNode($parent_code);
        }
    }

    private function addToRelationshipTable($code, $op_code, $parent_code, $type, $desc) : void
    {
        $status = $this->getValue('status');
        $values['code'] = $code;
        $values['op_code'] = $op_code;
        $values['parent_code'] = $parent_code;
        $values['type'] = $type;
        $values['desc'] = $desc;
        $values['node'] = 1;
        $values['status'] = $status;

        $this->response = (new DBQueries('tbl_operations_relationship', $values))->dbInsert();
    }

    private function setDeletedAtToNull($code) : void {
        DB::table('tbl_operations_relationship')->where('code', $code)->update(["deleted_at" => null]);
    }

    private function isExistingToTable($code) : int
    {
        $query = DB::table('tbl_operations_relationship')->where('code', $code)->first();

        if(!$query) return 0; // totally doesn't exist

        if(!is_null($query->deleted_at)) return -1; // exists but deleted

        return 1; // existing and not deleted
    }

    private function getOperationName($code) : string {
        $name = DB::table('tbl_operations')->where('code', $code)->value('name');
        return $name;
    }

    private function updateNode($parent_code) : void
    {
        DB::table('tbl_operations_relationship')->where('code', $parent_code)->update(["node" => 0]);
    }

    private function updateAsChildNode($parent_code) : void
    {
        DB::table('tbl_operations_relationship')->where('code', $parent_code)->whereNot('parent_code', '0')->update(["node" => 1]);
    }

    private function removeOperations()
    {
        // CHECK PARENT OF OPERATION TO DELETE
        // SEE WHETHER PARENT HAS CHILDREN
        // IF HAS OTHER CHILDREN, RETAIN NODE = 1
        // ELSE, UPDATE NODE AS PARENT = 0

        $code = $this->conditions['code'];

        if($this->isParent($code)) {
            return failedResponse("ERROR", ["msg" => 'Cannot delete parent node.'], 422);
        }

        $parent = $this->getParent($code);

        (new DBQueries('tbl_operations_relationship', null, conditions: ["code" => $code]))->dbDelete();

        if(!$this->hasChildren($parent)) {
            $this->updateAsChildNode($parent);
        }

        return true;
    }

    private function getParent($code) {
        $query = DB::table('tbl_operations_relationship')->select('parent_code')->where('code', $code)->first();
        return $query->parent_code;
    }

    private function isParent($code) : bool {
        $query = DB::table('tbl_operations_relationship')->select('node')->where('code', $code)->first();
        if($query->node === 0) return true;
        return false;
    }

    private function hasChildren($parent_code) : bool {
        $children = DB::table('tbl_operations_relationship')->where('parent_code', $parent_code)
            ->whereNull('deleted_at')->get();
        if($children->isEmpty()) return false;
        return true;
    }

    private function retrieveOperations() {
        // DOWN - TOP APPROACH

        // COMPARE CODE and get OP_CODE and its type PUT IN response[$type] => $OP_CODE
        // check if CODE HAS PARENT
        // - > YES => get OP_CODE of parent and type APPEND IN response[$type] => $OP_CODE
        // RECURSIVE, END IF NO PARENT
    }

    private function retrieveFromSubsidiaryLedger() : void
    {
        $code = $this->conditions['code'];
        $query = DB::table('tbl_subsidiary_ledger')->where('sl_code', $code)->whereNull('deleted_at')->get();
        if($query->isEmpty()) $this->isInSubsidiaryLedger = false;
        else $this->isInSubsidiaryLedger = true;
    }

    private function getResponse($action): ?JsonResponse
    {
        if (in_array($this->filterTable, array_keys(HIDDEN_RESPONSE_FIELDS))) $this->filterResponse();
        $responseMsg = "No Data Found!";
        if ($this->response) {
            if ($action == 'dbInsert') {
                $responseMsg = 'Successfully added!';
            } else if ($action === 'dbGetAll') {
                $response = $this->response;
                return successResponse("Success", $response);
            } else $responseMsg = "Success";
        }
        return successResponse($responseMsg, $this->response);
    }
}
