<?php

namespace Accounting\config;


use Accounting\ParamSetup;
use Accounting\setup\AccountMapping;
use Base\DBQueries;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class Entities extends ParamSetup
{
    private string $tbl = 'tbl_entities';
    private Request $request;
    private array $others;

    private $entityCode;
    private $entityType;
    private $entityStatus;
    private $subCategoryQuery;
    private $lastSubCategory;

    public function setParams(): array
    {
        return ['request' => $this->request, 'table' => $this->tbl];
    }

    public function main(Request $request): JsonResponse
    {
        $this->request = $request;

        try {
            if ($this->generateParams()) {
                $this->filterTable = $this->table;
                $action = $this->action;
                if (in_array($this->filterTable, array_keys(HIDDEN_REQUEST_FIELDS))) $this->others = $this->filterRequest();
                //$this->addValues('status', 1);
                $this->entityCode = $this->getValue('code');
                $this->entityType = $this->getValue("type");
                $this->entityStatus = $this->getValue('status');
                if($action === 'dbInsert' && !$this->checkDuplicates('tbl_entities', 'code', $this->entityCode)) {
                    return failedResponse("ERROR", ["msg" => 'Duplicate code.'], 422);
                }
                if (in_array($this->entityType, ["customer", "supplier"])) {
                    $tinNumber = data_get($this->others, "details.otherDetails.tinNumber");

                    if ($action === "dbInsert" && !$this->checkDuplicatesInDetails('tbl_entity_details', "tinNumber", $tinNumber, "otherDetails")) {
                        return failedResponse("ERROR", ["msg" => "Duplicate TIN Number."], 422);
                    }

                    $businessStyle = data_get($this->others, "details.otherDetails.businessStyle");

                    if ($action === "dbInsert" && !$this->checkDuplicatesInDetails('tbl_entity_details', "businessStyle", $businessStyle, "otherDetails")) {
                        return failedResponse("ERROR", ["msg" => "Duplicate Business Style."], 422);
                    }
                }
                $this->execQuery();
                if (in_array($this->action, ["dbInsert", "dbUpdate"]) && $this->request->hasFile("values.details.otherDetails.vatTaxFile")) $this->entityFileUpload(5);
                if ($this->response || $this->request->input("type") === "update") $this->saveOtherDetails();
                if ($this->response && $this->request->input("type") === "delete") $this->deleteDetails();
                //$this->updateAccountMapping();
                return $this->getResponse($action);
            }
        } catch (\Exception $ex) {
            return failedResponse("ERROR", ["msg" => $ex->getMessage()], 422);
        }
        return failedResponse("ERROR", ["msg" => "Unknown Error!"], 400);
    }

    private function getResponse($action): ?JsonResponse
    {
        if (in_array($this->filterTable, array_keys(HIDDEN_RESPONSE_FIELDS))) $this->filterResponse();
        $responseMsg = "No Data Found!";
        if ($this->response) {
            if ($action === 'dbInsert') {
                $responseMsg = "entity_code";
            }
            else if ($action === 'dbGet') {
                $response = $this->response;
                $res = new DBQueries("tbl_entity_details", null, ["entity_code" => $response->code]);
                $response->details = $res->dbGetAll();
                $this->entityParser($response);
                $response->billingDetails = $this->billingDetails($response);
                return successResponse("Success", $response);
            }
            else if ($action === 'dbGetAll') {
                $response = $this->response;
                foreach ($response as $key => $res) {
                    $res->details = (new DBQueries("tbl_entity_details", null, conditions: ["entity_code" => $res->code]))->dbGetAll();
                    $this->entityParser($res);
                }
                $response->map(function ($response){
                    $response->billingDetails = $this->billingDetails($response);
                });
                return successResponse("Success", $response);
            }
            else if($action === 'dbDelete') {
                $responseMsg = 'Successfully deleted.';
            }
            else $responseMsg = "Success";
        }
        return successResponse($responseMsg, $this->response);
    }
    public function billingDetails($response) : ?array {
        $billingDetails = array();
        $billingDetails['tin'] = null;
        $billingDetails['businessStyle'] = null;
        $billingDetails['email'] = null;
        $billingDetails['phone'] = null;
        $billingDetails['address'] = null;
        if(isset($response->otherDetails)){
            $otherDetails = $response->otherDetails;
            $billingDetails['tin'] = $otherDetails['tinNumber'];
            $billingDetails['businessStyle'] = $otherDetails['businessStyle'];
            $billingDetails['email'] = $otherDetails['email'];
        }
        if(isset($response->billingAddress)){
            $billingDetails['phone'] = $response->billingAddress['phone'];
            $billingDetails['address'] = $this->addressParser($response->billingAddress);
        }
        return $billingDetails;
    }
    private function addressParser($addresses) : ?string {
        $province = DB::connection('sys_base')->table('tbl_addresses')->where('type', 'province')->where('code',$addresses['provinceCode'])->first()->name;
        $city = DB::connection('sys_base')->table('tbl_addresses')->where('type', 'city')->where('code',$addresses['cityCode'])->first()->name;
        $brgy = DB::connection('sys_base')->table('tbl_addresses')->where('type', 'barangay')->where('code',$addresses['barangayCode'])->first()->name;
        return ucwords(strtolower($brgy.', '.$city.', '.$province));
    }

    public function entityParser($response) {
        foreach($response->details as $detail) {
            $key = $detail->category;
            if (is_null($detail->sub_category)) {
                $response->$key[$detail->type] = $detail->value ?? "";
            } else {
                if (!property_exists($response, $key)) $response->$key = [];
                if (!array_key_exists($detail->sub_category, $response->$key)) $response->$key[$detail->sub_category] = [];
                $response->$key[$detail->sub_category][$detail->type] = $detail->value ?? "";
            }
        }
        unset($response->details);
        return $response;
    }

    private function saveOtherDetails()
    {
        if ($this->action === 'dbInsert') {
            $details = $this->others;
            //return successResponse("", $details);
            foreach ($details['details'] as $k => $detail) {
                if (is_array($detail) || is_object($detail)) {
                    foreach ($detail as $key => $value) {
                        if (is_array($value)) {
                            foreach ($value as $k2 => $val) {
                                $this->addOtherDetails($k2, $k, $val, $key);
                            }
                        } else $this->addOtherDetails($key, $k, $value);
                    }
                } else $this->addOtherDetails($k, 'info', $detail);
            }
        } else if ($this->action === "dbUpdate") {
            DB::table('tbl_entity_details')->where('entity_code', $this->entityCode)
            ->update(['status' => $this->entityStatus]);
            $details = $this->others;
            foreach($details['details'] as $k => $detail) {
                if (is_array($detail)) {
                    foreach($detail as $key => $value) {
                        if(is_int($key)) {
                            $this->lastSubCategory = $key;
                        }

                        if (is_array($value)) {
                            foreach ($value as $k2 => $val) {
                                $this->subCategoryQuery = DB::table('tbl_entity_details')
                                ->where('entity_code', $this->entityCode)
                                ->where('category', $k)
                                ->where('type', $k2)
                                ->orderByDesc('sub_category')
                                ->first();
                                $this->updateOtherDetails($k2, $k, $val, $key);
                            }
                        }else  {
                            $this->subCategoryQuery = DB::table('tbl_entity_details')->where('entity_code', $this->entityCode)
                            ->where("type", $key)
                            ->where('category', $k)->orderByDesc('sub_category')->first();
                            $type = DB::table('tbl_entity_details')->where('entity_code', $this->entityCode)->where('type', $key)->get();
                            if($type->isEmpty()) $this->addOtherDetails($key, $k, $value);
                            else $this->updateOtherDetails($key, $k, $value);
                        }
                    }
                } else {
                    // caters to adding new detail type for existing data on update
                    $type = DB::table('tbl_entity_details')->where('entity_code', $this->entityCode)->where('type', $k)->get();
                    if($type->isEmpty()) {
                        $this->addOtherDetails($k, 'info', $detail);
                    } else {
                        $this->updateNonArray($k, "info", $detail);
                    }
                }
                $this->onUpdateRemoveFields($k, $this->lastSubCategory);
            }
        }
    }

    private function deleteDetails() : void {
        $conditions = ["entity_code" => $this->conditions["code"]];
        $this->response = (new DBQueries("tbl_entity_details", conditions: $conditions))->dbDelete();
    }

    private function updateAccountMapping()
    {
        $mapper = [
            "customer" => [
                "normalBalance" => "Dr",
                "category" => "receivable"
            ],
            "supplier" => [
                "normalBalance" => "Cr",
                "category" => "payable"
            ]
        ];
        $accountCode = data_get($this->others, "details.otherDetails.accountCode");

        $accountMappingRecord = (new AccountMapping)->getAccountMapRecord($this->entityCode);
        if (is_null($accountMappingRecord) && !is_null($accountCode) && in_array($this->entityType, array_keys($mapper))) {
            (new AccountMapping)->saveAccountMapRecord($accountCode, $this->entityCode, $mapper[$this->entityType]["normalBalance"], "billing", $mapper[$this->entityType]["category"]);
        } else if (!is_null($accountCode) && in_array($this->entityType, array_keys($mapper)) && $this->request->input("type") === "update") {
            (new AccountMapping)->updateAccountMapRecord($accountCode, $this->entityCode, $mapper[$this->entityType]["normalBalance"], "billing", $mapper[$this->entityType]["category"]);
        } else if (!is_null($accountCode) && $this->request->input("type") === "delete") {
            (new AccountMapping)->deleteAccountMapRecord($this->entityCode);
        }
    }

    private function updateOtherDetails($type, $category, $value, $sub_category = null): void
    {
        $conditions["entity_code"] = $this->entityCode;
        $conditions["type"] = $type;
        $conditions["category"] = $category;
        if (!is_null($sub_category)) $conditions["sub_category"] = $sub_category;
        $values["value"] = $value;
        $values["status"] = $this->entityStatus;

        if (is_null($this->subCategoryQuery)) {
            $this->addOtherDetails($type, $category, $value, $sub_category);
            return;
        }

        if($this->subCategoryQuery && !is_null($this->subCategoryQuery->sub_category)) {
            if($sub_category > $this->subCategoryQuery->sub_category) {
                $this->addOtherDetails($type, $category, $value, $sub_category);
                return;
            }
        }

        $this->response = (new DBQueries("tbl_entity_details", $values, $conditions))->dbUpdate();
    }

    private function updateNonArray($type, $category, $value): void
    {
        $conditions["entity_code"] = $this->entityCode;
        $conditions["type"] = $type;
        $conditions["category"] = $category;
        $values["value"] = $value;
        $values["status"] = $this->entityStatus;

        $this->response = (new DBQueries("tbl_entity_details", $values, $conditions))->dbUpdate();
    }

    private function addOtherDetails($type, $category, $value, $sub_category = null): void
    {
        $values['entity_code'] = $this->entityCode;
        $values['type'] = $type;
        $values['category'] = $category;
        if (!is_null($sub_category)) $values["sub_category"] = $sub_category;
        $values['value'] = $value;
        $values['status'] = $this->entityStatus;
        $add = new DBQueries('tbl_entity_details', $values);
        $this->response = $add->dbInsert();
    }

    private function entityFileUpload($nameSpaceId) {
        $record = DB::connection("sys_base")->table("namespace")->where("id", $nameSpaceId)->first();
        if ($this->request->hasFile("values.details.otherDetails.vatTaxFile") && $this->request->file("values.details.otherDetails.vatTaxFile")->isValid()) {
            $clientFile = $this->request->file("values.details.otherDetails.vatTaxFile");
            $file = fileUpload($this->request->file("values.details.otherDetails.vatTaxFile"),
                $clientFile->getClientOriginalName(),
                str_replace(".", "/", $record->namespace)
            );

            data_set($this->others, "details.otherDetails.vatTaxFile", $file);
        }
    }

    private function onUpdateRemoveFields($category, $lastSubCategory) : void
    {
        $getData = DB::table('tbl_entity_details')->where('entity_code', $this->entityCode)->where('category', $category)->get();
        if(!is_null($getData->value('sub_category'))) {
            foreach ($getData as $data) {
                if($data->sub_category > $lastSubCategory) {
                    DB::table('tbl_entity_details')->where('entity_code', $this->entityCode)->where('category', $category)
                    ->where('type', $data->type)->where('value', $data->value)
                    ->where('sub_category', $data->sub_category)->delete();
                }
            }
        }
    }
}
